<?php

// Normal API (For other panels like: HexaSoftwareTech, NidukaAkalanka, FranzKafkaYu ...)

class XuiApi
{
    protected $https, $port, $url, $username, $password, $id, $cookie, $client;
    protected $path = [
        'login' => '/login',
        'status' => '/server/status',
        'restartXrayService' => '/server/restartXrayService',
        'stopXrayService' => '/server/stopXrayService',
        'restartPanel' => '/xui/setting/restartPanel',
        'getXrayVersion' => '/server/getXrayVersion',
        'installXray' => 'server/installXray/{id}',
        'allSetting' => '/xui/setting/all',
        'updateSetting' => '/xui/setting/update',
        'updateUser' => '/xui/setting/updateUser',
        'listInbound' => '/xui/inbound/list',
        'inbound' => '/xui/inbound/get/{id}',
        'delInbound' => '/xui/inbound/del/{id}',
        'updateInbound' => '/xui/inbound/update/{id}',
        'addInbound' => '/xui/inbound/add',
        'resetClientTraffic' => '/xui/inbound/resetClientTraffic/{client}',
    ];
    protected $defaults = [
        'sniffing' => [
            "enabled" => true,
            "destOverride" => [
                "http",
                "tls"
            ]
        ],
    ];

    public function __construct($https, $url, $port, $username, $password)
    {
        $this->https = $https;
        $this->port = $port;
        $this->url = $url;
        $this->cookie = __DIR__ . DIRECTORY_SEPARATOR . "$url.cookie.txt";
        $this->username = $username;
        $this->password = $password;
        if (!file_exists($this->cookie)) {
            $this->login();
        }
    }

    protected function curl($path, $body = [], $isPost = true)
    {
        $curl = curl_init();
        $error = false;
        curl_setopt_array($curl, array(
            CURLOPT_URL => $this->getUrl($path),
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_COOKIEFILE => $this->getCookie(),
            CURLOPT_COOKIEJAR => $this->getCookie(),
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $isPost == true ? 'POST' : 'GET',
            CURLOPT_POSTFIELDS => http_build_query($body),
        ));
        $res = json_decode(curl_exec($curl), true);
        if (curl_errno($curl) || !$res) $error = true;
        curl_close($curl);
        if ($error) unlink($this->cookie);
        return $res;
    }

    protected function setId($id)
    {
        $this->id = $id;
        return $this;
    }

    protected function getId(): int
    {
        return $this->id;
    }

    protected function setClient($client)
    {
        $this->client = $client;
        return $this;
    }

    protected function getClient()
    {
        return $this->client;
    }

    protected function getUrl($path): string
    {

        if (isset($this->path[$path])) {
            $urlPath = $this->path[$path];
            $arrPath = ['delInbound', 'inbound', 'updateInbound', 'installXray', 'delClient', 'clientIps', 'clearClientIps'];
            $arrPathWithClient = ['resetClientTraffic'];
            if (in_array($path, $arrPath)) {
                $urlPath = strtr($this->path[$path], ['{id}' => $this->getId()]);
            }
            if (in_array($path, $arrPathWithClient)) {
                $urlPath = strtr($this->path[$path], ['{client}' => $this->getClient()]);
            }
            return $this->https ? "https://{$this->url}:{$this->port}" . $urlPath : "http://{$this->url}:{$this->port}"  . $urlPath;
        }

        return $this->https ? "https://{$this->url}:{$this->port}" : "http://{$this->url}:{$this->port}";
    }


    public function getCookie()
    {
        return $this->cookie;
    }

    protected function resetCookieFile()
    {
        unlink($this->cookie);
        return $this;
    }


    protected function auth()
    {
        $this->curl('login', ['username' => $this->username, 'password' => $this->password]);

        return $this;
    }

    public function login()
    {
        $this->auth();
        return $this;
    }

    public function status()
    {
        return $this->curl('status', [], true);
    }

    public function getXrayVersion()
    {
        return $this->curl('getXrayVersion', [], true);
    }

    public function installXray($version = 'v1.6.4')
    {
        $this->setId($version);
        return $this->curl('installXray', compact('version'), true);
    }

    public function restartXrayService()
    {
        return $this->curl('restartXrayService', [], true);
    }

    public function restartPanel()
    {
        return $this->curl('restartPanel', [], true);
    }

    public function stopXrayService()
    {
        return $this->curl('stopXrayService', [], true);
    }

    public function startXrayService()
    {
        return $this->curl('restartXrayService', [], true);
    }


    public function listInbound()
    {
        return $this->curl('listInbound', [], true);;
    }

    public function allSetting()
    {
        return $this->curl('allSetting', [], true);
    }


    public function updateUser($oldUsername, $oldPassword, $newUsername, $newPassword)
    {
        return $this->curl('updateUser', compact('oldPassword', 'oldUsername', 'newPassword', 'newUsername'), true);
    }

    public static function convertFileSize($size, $from, $to)
    {
        $units = array('B', 'KB', 'MB', 'GB', 'TB', 'PB');
        $from_index = array_search($from, $units);
        $to_index = array_search($to, $units);

        if ($from_index === false || $to_index === false)
            return "Invalid units";


        if (!is_numeric($size) || $size < 0)
            return "Invalid size";


        return $size * pow(1024, $from_index - $to_index);
    }

    public function getClientIndex($clients, $uid)
    {
        if (count($clients) == 0)
            return false;
        $i = 0;
        foreach ($clients as $item) {
            $cuid = $item['password'] ?? $item['id'];
            if ($uid == $cuid)
                return (int)$i;
            $i++;
        }
        return false;
    }

    public function getClientIndexByEmail($clients, $email)
    {
        if (count($clients) == 0)
            return false;
        $i = 0;
        foreach ($clients as $item) {
            if ($email == $item['email'])
                return (int)$i;
            $i++;
        }
        return false;
    }

    public function getClientStatsIndexByEmail($clients, $email)
    {
        if (count($clients) == 0)
            return false;
        $i = 0;
        foreach ($clients as $item) {
            if ($email == $item['email'])
                return (int)$i;
            $i++;
        }
        return false;
    }

    public function checkExistsUuidOnClients($clients, $uid)
    {
        if (count($clients) == 0)
            return false;
        foreach ($clients as $item) {
            $cuid = $item['password'] ?? $item['id'];
            if ($uid == $cuid)
                return true;
        }
        return false;
    }


    public function list(array $filter = [])
    {
        $list = $this->listInbound();
        if (!isset($list['obj'])) return false;
        if ($filter == []) return $list;
        $result = [];
        foreach ($list['obj'] as $item) {
            $status = true;
            $settings = json_decode($item["settings"], true);

            if (!empty($filter["id"]) && $filter["id"] !== (int)$item['id']) $status = false;
            if (!empty($filter["port"]) && $filter["port"] !== (int)$item['port']) $status = false;
            if (!empty($filter["uid"]) && !$this->checkExistsUuidOnClients($settings['clients'], $filter["uid"])) $status = false;
            if (!empty($filter["protocol"]) && $filter["protocol"] !== $item['protocol']) $status = false;
            if ($status)
                $result[] = $item;
        }
        return $result;
    }

    public function editInbound($enable, $id, $remark, $port, $protocol, $settings, $streamSettings, $total = 0, $up = 0, $down = 0, $sniffing = null, $expiryTime = 0, $listen = '')
    {
        $sniffing = $sniffing == null ? $this->defaults['sniffing'] : $sniffing;
        $sniffing =  json_encode($sniffing, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $settings =  json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $streamSettings =  json_encode($streamSettings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $this->setId($id);
        return $this->curl('updateInbound', compact('enable', 'remark', 'port', 'protocol', 'settings', 'streamSettings', 'total', 'up', 'down', 'sniffing', 'expiryTime', 'listen'), true);
    }

    public function removeClientByEmail($id, $Email)
    {
        $list = $this->list(['id' => (int)$id])[0];
        $enable = (bool)$list['enable'];
        $remark = $list['remark'];
        $port = $list['port'];
        $protocol = $list['protocol'];
        $settings = json_decode($list["settings"], true);
        $cIndex = $this->getClientIndexByEmail($settings['clients'], $Email);
        if ($cIndex === false)
            return false;
        unset($settings['clients'][$cIndex]);
        $streamSettings = json_decode($list['streamSettings']);
        $up = $list['up'];
        $down = $list['down'];
        $sniffing = json_decode($list['sniffing']);
        $expiryTime = $list['expiryTime'];
        $listen = $list['listen'];
        $total = $list['total'];
        return $this->editInbound($enable, $id, $remark, $port, $protocol, $settings, $streamSettings, $total, $up, $down, $sniffing, $expiryTime, $listen);
    }

    public function removeInbound($id, $email = '')
    {
        if ($email == '') {
            $this->setId($id);
            return $this->curl('delInbound', [], true);
        } else {
            return $this->removeClientByEmail($id, $email);
        }
    }

    public function updateSetting($webPort, $webCertFile, $webKeyFile, $webBasePath, $xrayTemplateConfig, bool $tgBotEnable = false, $tgNotifyExpireTimeDiff = 0, $tgNotifyTrafficDiff = 0, string $tgBotToken = null, $tgBotChatId = null, $tgRunTime = null, $timeLocation = 'Asia/Tehran', $webListen = '')
    {
        $com = compact('webPort', 'webCertFile', 'webKeyFile', 'webBasePath', 'xrayTemplateConfig', 'tgBotEnable', 'tgNotifyExpireTimeDiff', 'tgNotifyTrafficDiff', 'tgBotToken', 'tgBotChatId', 'tgRunTime', 'timeLocation', 'webListen');
        return $this->curl('updateSetting', $com, true);
    }
    private function guidv4($data = null)
    {
        $data = $data ?? random_bytes(16);
        assert(strlen($data) == 16);
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40);
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
    private function randomPassword()
    {
        $alphabet = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890';
        $pass = array();
        $alphaLength = strlen($alphabet) - 1;
        for ($i = 0; $i < 8; $i++) {
            $n = rand(0, $alphaLength);
            $pass[] = $alphabet[$n];
        }
        return implode($pass);
    }

    // generating Vless link
    private function generateVlessLink($guidv4, $port, $remark, $network = 'ws', $security = 'none', $headerType = '', $host = '')
    {
        $params = [];
        $params['type'] = $network;
        $params['security'] = $security;
        switch ($params['type']) {
            case "ws":
                $params["path"] = "/";
                break;

            case "tcp":
                $params["path"] = "/";
                if ($headerType == "http") $params["headerType"] = "http";
                if ($host) $params["host"] = $host;
                break;
        }
        switch ($security) {
            case "tls":
                $params['sni'] = "DOMAIN";
                break;

            case "xtls":
                $params['flow'] = "";
                break;
        }
        $link = "vless://$guidv4@URL:$port?";
        $query = http_build_query($params);
        $link = $link . $query . "#" . $remark;
        return $link;
    }

    // generating Vmess link
    private function generateVmessLink($guidv4, $port, $remark, $network = 'ws', $security = 'none', $headerType = '', $host = '')
    {

        $add = $security != 'none' ? "DOMAIN" : "URL";
        switch ($network) {
            case "ws":
                $path = "/";
                break;
            case "tcp":
                $path = "/";
                break;
            default:
                $path = "";
                break;
        }
        $tls =  $network == 'tls' ? 'tls' : 'none';
        $type = 'none';
        if ($headerType == 'http') $type = 'http';

        $link = 'vmess://' . base64_encode('{
"v": "2",
"ps": "' . $remark . '",
"add": "' . $add . '",
"port": ' . $port . ',
"id": "' . $guidv4 . '",
"aid": 0,
"net": "' . $network . '",
"type": "' . $type . '",
"tls": "' . $tls . '",
"path": "' . $path . '",
"host": "' . $host . '"
        }');
        return $link;
    }


    // generating Trojan Link
    private function generateTrojanLink($password, $port, $remark)
    {
        $link = "trojan://$password@URL:$port#$remark";
        return $link;
    }


    // Add inbound
    public function addInbound($total, $up, $down, $enable, $remark, $port, $protocol, $expiryTime = 0, $limitIp = 0, $network = 'ws', $security = 'none', $domain = '', $host = '', $listen = '')
    {
        $guidv4 = $this->guidv4();
        if ($port == null) $port = rand(10000, 60000);
        $headerType = '';
        if ($network == 'tcp')  $headerType = 'http';
        switch ($protocol) {
            case "vmess":
                $link = $this->generateVmessLink($guidv4, $port, $remark, $network, $security, $headerType, $host);
                $settings = [
                    "clients" => [
                        [
                            "id" => $guidv4,
                            "alterId" => 0,
                            "email" => "",
                            "limitIp" => 1,
                            "totalGB" => 0,
                            "expiryTime" => ""
                        ]
                    ],
                    "disableInsecureEncryption" => false
                ];

                $streamSettings = [];

                break;

            case "trojan":
                $password = $this->randomPassword();
                $link = $this->generateTrojanLink($password, $port, $remark);
                $settings = [
                    "clients" => [
                        [
                            "password" => $password,
                            "flow" => "xtls-rprx-direct"
                        ]
                    ],
                    "fallbacks" => []
                ];

                $streamSettings = [];
                break;

            case "vless":
                $link = $this->generateVlessLink($guidv4, $port, $remark, $network, $security, $headerType, $host);
                $settings = [
                    "clients" => [
                        [
                            "id" => $guidv4,
                            "flow" => "xtls-rprx-direct",
                            "email" => "",
                            "limitIp" => $limitIp,
                            "totalGB" => 0,
                            "expiryTime" => ""

                        ],
                    ],
                    "decryption" => "none",
                    "fallbacks" => []
                ];
                $streamSettings = [];
                break;
        }

        switch ($network) {
            case "ws":
                $streamSettings['network'] = 'ws';
                $streamSettings['wsSettings'] = [
                    "acceptProxyProtocol" => false,
                    "path" => "/",
                    "headers" => [
                        "Host" => [
                            $host
                        ]
                    ]
                ];
                if ($host == '') $streamSettings['wsSettings']['headers'] = (object) array();
                break;
            case "grpc":
                $streamSettings['network'] = 'grpc';
                $streamSettings['grpcSettings'] = [
                    "serviceName" => ""
                ];
                break;
            case "tcp":
                $streamSettings['network'] = 'tcp';
                $streamSettings['tcpSettings'] = [
                    "acceptProxyProtocol" => false,
                    "header" => [
                        "type" => "http",
                        "request" => [
                            "method" => "GET",
                            "path" => ["/"],
                            "headers" => [
                                "Host" => [
                                    $host
                                ]
                            ]
                        ],
                        "response" => [
                            "version" => "1.1",
                            "status" => "200",
                            "reason" => "OK",
                            "headers" => [
                                "Content-Type" => [
                                    "application/octet-stream"
                                ]
                            ]

                        ]

                    ],
                ];
                if ($host == '') $streamSettings['tcpSettings']['header']['request']['headers'] = (object) array();

                break;

            default:
                $streamSettings['network'] = 'ws';
                $streamSettings['wsSettings'] = [
                    "acceptProxyProtocol" => false,
                    "path" => "/",
                    "headers" => [
                        "Host" => [
                            $host
                        ]
                    ]
                ];
                if ($host == '') $streamSettings['wsSettings']['headers'] = (object) array();
                break;
        }

        switch ($security) {
            case "tls":
                $streamSettings["security"] = "tls";
                $streamSettings["tlsSettings"] = [
                    "serverName" => $domain,
                    "certificates" => [
                        [
                            "certificateFile" => "/root/cert.crt",
                            "keyFile" => "/root/private.key"
                        ]
                    ],
                    "alpn" => []
                ];
                break;

            case "xtls":
                $streamSettings["security"] = "xtls";
                $streamSettings["xtlsSettings"] = [
                    "serverName" => $domain,
                    "certificates" => [
                        [
                            "certificateFile" => "/root/cert.crt",
                            "keyFile" => "/root/private.key"
                        ]
                    ],
                    "alpn" => []
                ];
                break;
            case "none":
                $streamSettings["security"] = "none";
                break;
            default:
                $streamSettings["security"] = "none";
                break;
        }

        $sniffing =  $this->defaults['sniffing'];
        $sniffing =  json_encode($sniffing, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $settings =  json_encode($settings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $streamSettings =  json_encode($streamSettings, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        $data = $this->curl('addInbound', compact('total', 'up', 'down', 'enable', 'remark', 'port', 'protocol', 'settings', 'streamSettings', 'sniffing', 'expiryTime', 'listen'), true);
        @$newData = [
            "success" => $data['success'],
            "id" => $data['obj']['id'],
            "port" => $data['obj']['port'],
            "protocol" => $data['obj']['protocol'],
            "link" => $link,
        ];
        return $newData;
    }


    // add client to inbound (FOR MUTLTI CLIENT ON 1 PORT)
    public function addClientToInbound($id, $total, $protocol, $remark, $email, $expiryTime = 0, $tunnel = '0', $limitIp = 0, $host = '', $domain = '')
    {
        $guidv4 = $this->guidv4();
        $list = $this->list(['id' => (int)$id])[0];
        $enable = (bool)$list['enable'];
        // $remark = $list['remark'];
        $port = $list['port'];
        $protocol = $list['protocol'];
        $settings = json_decode($list["settings"], true);
        $streamSettings = json_decode($list['streamSettings']);

        $domain =  '';
        $host = '';
        $headerType = '';
        if (@$streamSettings->{'tlsSettings'}->{'serverName'}) {
            $domain = $streamSettings->{'tlsSettings'}->{'serverName'};
        }
        if (@$streamSettings->{'xtlsSettings'}->{'serverName'}) {
            $domain = $streamSettings->{'xtlsSettings'}->{'serverName'};
        }
        if (@$streamSettings->{'tcpSettings'}->{'header'}->{'request'}->{'headers'}->{'Host'}[0]) {
            $host = $streamSettings->{'tcpSettings'}->{'header'}->{'request'}->{'headers'}->{'Host'}[0];
        }
        if (@$streamSettings->{'wsSettings'}->{'headers'}->{'Host'}) {
            $host = $streamSettings->{'wsSettings'}->{'headers'}->{'Host'};
        }
        if (@$streamSettings->{'tcpSettings'}->{'header'}->{'type'}) {
            $headerType = $streamSettings->{'tcpSettings'}->{'header'}->{'type'};
        }
        $newClient =
            [
                "id" => $guidv4,
                "alterId" => 0,
                "email" => $email,
                "limitIp" => $limitIp,
                "totalGB" => $total,
                "expiryTime" => $expiryTime
            ];
        switch ($protocol) {
            case "vmess":
                $link = $this->generateVmessLink($guidv4, $list['port'], $remark, $streamSettings->network, $streamSettings->security, $headerType, $host);
                $newClient =
                    [
                        "id" => $guidv4,
                        "alterId" => 0,
                        "email" => $email,
                        "limitIp" => $limitIp,
                        "totalGB" => $total,
                        "expiryTime" => $expiryTime
                    ];
                break;
            case "vless":
                $link = $this->generateVlessLink($guidv4,  $list['port'], $remark, $streamSettings->network, $streamSettings->security, $headerType, $host);
                $newClient =
                    [
                        "id" => $guidv4,
                        "flow" => "xtls-rprx-direct",
                        "email" => $email,
                        "limitIp" => $limitIp,
                        "totalGB" => $total,
                        "expiryTime" => $expiryTime

                    ];
                break;
        }

        $settings['clients'][] =  $newClient;
        $up = $list['up'];
        $down = $list['down'];
        $sniffing = json_decode($list['sniffing']);
        $expiryTime = $list['expiryTime'];
        $listen = $list['listen'];
        $total = $list['total'];
        $remark = $list['remark'];
        $data =  $this->editInbound($enable, $id, $remark, $port, $protocol, $settings, $streamSettings, $total, $up, $down, $sniffing, $expiryTime, $listen);
        @$newData = [
            "success" => $data['success'],
            "id" => $data['obj']['id'],
            "port" => $data['obj']['port'],
            "protocol" => $data['obj']['protocol'],
            "link" => $link,
            "domain" => $domain
        ];
        return  $newData;
    }

    // SECTION #5

    // reset Traffic and expire date
    public function resetTrafficAndExpire($id, $expiryTime, $email = '')
    {
        $list = $this->list(['id' => (int)$id])[0];
        $enable = (bool)$list['enable'];
        $remark = $list['remark'];
        $port = $list['port'];
        $protocol = $list['protocol'];
        $settings = json_decode($list["settings"], true);
        if ($email) {
            $cIndex = $this->getClientIndexByEmail($settings['clients'], $email);
            if ($cIndex === false)
                return false;
            $settings['clients'][$cIndex]['expiryTime'] = $expiryTime;
            $up = $list['up'];
            $down = $list['down'];
            $expiryTime = $list['expiryTime'];
            // $this->resetClientTraffic($email);
        } else {
            $down = 0;
            $up = 0;
            $expiryTime = $expiryTime;
        }
        $streamSettings = json_decode($list['streamSettings']);
        $sniffing = json_decode($list['sniffing']);
        $listen = $list['listen'];
        $total = $list['total'];
        return $this->editInbound($enable, $id, $remark, $port, $protocol, $settings, $streamSettings, $total, $up, $down, $sniffing, $expiryTime, $listen);
    }


    // Increase Config traffic
    public function increaseTraffic($id, $traffic, $email = '')
    {

        $list = $this->list(['id' => (int)$id])[0];
        $enable = (bool)$list['enable'];
        $remark = $list['remark'];
        $port = $list['port'];
        $protocol = $list['protocol'];
        $settings = json_decode($list["settings"], true);
        $total = $list['total'];
        if ($email) {
            $cIndex = $this->getClientIndexByEmail($settings['clients'], $email);
            if ($cIndex === false)
                return false;
            $settings['clients'][$cIndex]['totalGB'] = $settings['clients'][$cIndex]['totalGB'] + $this->convertFileSize($traffic, 'GB', 'B');
        } else {
            $total += $this->convertFileSize($traffic, 'GB', 'B');
        }
        $streamSettings = json_decode($list['streamSettings']);
        $up = $list['up'];
        $down = $list['down'];
        $sniffing = json_decode($list['sniffing']);
        $expiryTime = $list['expiryTime'];
        $listen = $list['listen'];

        return $this->editInbound($enable, $id, $remark, $port, $protocol, $settings, $streamSettings, $total, $up, $down, $sniffing, $expiryTime, $listen);
    }

    // change service link - by changing UID
    public function changeLink($id, $remark, $email = '', $host = '')
    {
        $guidv4 = $this->guidv4();

        $list = $this->list(['id' => (int)$id])[0];
        $enable = (bool)$list['enable'];
        $port = $list['port'];
        $protocol = $list['protocol'];
        $settings = json_decode($list["settings"], true);
        $total = $list['total'];
        $streamSettings = json_decode($list['streamSettings']);
        $up = $list['up'];
        $down = $list['down'];
        $sniffing = json_decode($list['sniffing']);
        $expiryTime = $list['expiryTime'];
        $listen = $list['listen'];
        $newHost = '';
        if (!$host) {
            if (@$streamSettings->{'tcpSettings'}->{'header'}->{'request'}->{'headers'}->{'Host'}[0]) {
                $newHost = $streamSettings->{'tcpSettings'}->{'header'}->{'request'}->{'headers'}->{'Host'}[0];
            }
            if (@$streamSettings->{'wsSettings'}->{'headers'}->{'Host'}) {
                $newHost = $streamSettings->{'wsSettings'}->{'headers'}->{'Host'};
            }
        }
        $headerType = '';
        if (@$streamSettings->{'tcpSettings'}->{'header'}->{'type'}) {
            $headerType = $streamSettings->{'tcpSettings'}->{'header'}->{'type'};
        }
        if ($email == '') {
            $settings['clients'][0]['id'] =  $guidv4;
        } else {
            $cIndex = $this->getClientIndexByEmail($settings['clients'], $email);
            if ($cIndex === false)
                return false;
            $settings['clients'][$cIndex]['id'] = $guidv4;
        }
        $link = '';
        $host = $newHost ? $newHost : 'HOST';
        switch ($protocol) {
            case "vless":
                $link = $this->generateVlessLink($guidv4, $port, $remark !== 'none' ? $remark : $list['remark'], $streamSettings->network, $streamSettings->security, $headerType, $host);
                break;
            case "vmess":
                $link = $this->generateVmessLink($guidv4, $port, $remark !== 'none' ? $remark : $list['remark'], $streamSettings->network, $streamSettings->security, $headerType, $host);
                break;
        }

        $data = $this->editInbound($enable, $id, $list['remark'], $port, $protocol, $settings, $streamSettings, $total, $up, $down, $sniffing, $expiryTime, $listen);

        @$newData = [
            "success" => $data['success'],
            "id" => $data['obj']['id'],
            "port" => $data['obj']['port'],
            "protocol" => $data['obj']['protocol'],
            "link" => $link,
        ];
        return $newData;
    }


    public function resetClientTraffic($client)
    {
        $this->setClient($client);
        return $this->curl('resetClientTraffic');
    }
}
